﻿using System;
using Hims.Shared.UserModels.Common;
using Hims.Shared.UserModels.GataPass;

namespace Hims.Api.Controllers
{
    using System.Threading.Tasks;

    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/gatePass")]
    public class GatePassController : BaseController
    {
        /// <summary>
        /// The service
        /// </summary>
        private readonly IGatePassService service;

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IPatientService patientServices;

        /// <inheritdoc />
        public GatePassController(IGatePassService service, IAuditLogService auditLogServices, IPatientService patientServices)
        {
            this.service = service;
            this.auditLogServices = auditLogServices;
            this.patientServices = patientServices;
        }

       
        /// <summary>
        /// Inserts the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] InsertModel model)
        {
            try
            {              
                var response = await this.service.FetchAsync(model);
              
                return Ok(new GenericResponse
                {
                    Status = response != null ? GenericStatus.Success : GenericStatus.Info,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// Inserts the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] InsertModel model,[FromHeader]LocationHeader header)
        {
            model.CreatedDate = DateTime.Now;
            var response = await this.service.InsertAsync(model);
            var patientFullName = await this.patientServices.FindPatientByAdmissionId(model.AdmissionId);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Admission,
                    LogFrom = (short)model.RoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $"<b>{model.FullName}</b> has <b>Created GATE Pas</b> for  the Patient <b>{patientFullName}</b> on <b>{DateTime.UtcNow.AddMinutes(330)}</b> after Generating Final Bill.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }

            return Ok(new GenericResponse
            {
                Status = response > 0 ? GenericStatus.Success : GenericStatus.Error,
                Data = response
            });
            
        }

        /// <summary>
        /// Inserts the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] InsertModel model,[FromHeader] LocationHeader header)
        {
            model.ModifiedDate = DateTime.Now;
            var response = await this.service.UpdateAsync(model).ConfigureAwait(false);
            var patientFullName = await this.patientServices.FindPatientByAdmissionId(model.AdmissionId);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Admission,
                    LogFrom = (short)model.RoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $"<b>{model.FullName}</b> has <b>updated</b> the GatePass for the Patient <b>{patientFullName}</b>  on <b>{DateTime.UtcNow.AddMinutes(330)}</b>",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

            }
            return Ok(new GenericResponse
            {
                Status = response > 0 ? GenericStatus.Success : GenericStatus.Error,
                Data = response
            });
            
        }

 }
}